<?php
/**
 * Thrive Themes - https://thrivethemes.com
 *
 * @package thrive-apprentice
 */

namespace TVA\Buy_Now;

use Exception;
use TVA\Stripe\Connection;
use function get_home_url;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Silence is golden!
}

class Stripe extends Generic {

	public static $CACHE = [];


	/**
	 * @var mixed|string
	 */
	private $product_id;
	private $price_id;
	private $success_url;
	private $cancel_url;
	private $test_mode;
	private $price_type;
	private $stripe_connection;

	public function __construct( $data ) {
		parent::__construct( $data );
		$this->parse_data();
	}

	protected function parse_data() {
		$this->price_id = isset( $this->data['price_id'] ) ? $this->data['price_id'] : '';
		if ( ! empty( $this->price_id ) ) {
			$this->product_id = isset( $this->data['product_id'] ) ? $this->data['product_id'] : '';
			$this->test_mode  = isset( $this->data['test_mode'] ) ? $this->data['test_mode'] : false;

			$price_type_key   = $this->test_mode ? 'test_price_type' : 'live_price_type';
			$this->price_type = isset( $this->data[ $price_type_key ] ) ? $this->data[ $price_type_key ] : '';


			$this->success_url = isset( $this->data['success_url'] ) ? $this->data['success_url'] : get_home_url();
			$this->cancel_url  = isset( $this->data['cancel_url'] ) ? $this->data['cancel_url'] : '';

			if ( ! empty( $this->success_url ) ) {
				$this->success_url = $this->append_url( $this->success_url );
			}
			if ( ! empty( $this->cancel_url ) ) {
				$this->cancel_url = $this->append_url( $this->cancel_url );
			}

			$stripe = Connection::get_instance();
			$stripe->set_test_mode( $this->test_mode );
			$this->stripe_connection = $stripe->get_client();
		}
	}

	public function is_valid() {
		return ! empty( $this->price_id );
	}

	/**
	 * Get the URL for the checkout session
	 *
	 * @return string
	 */
	public function get_url() {
		if ( $this->price_id && ! isset( static::$CACHE[ $this->price_id ] ) ) {
			$checkout_data = [
				'mode'       => $this->price_type === 'recurring' ? 'subscription' : 'payment',
				'line_items' => [
					[
						'price'    => $this->price_id,
						'quantity' => 1,
					],
				],
			];

			if ( ! empty( $this->success_url ) ) {
				$checkout_data['success_url'] = $this->success_url;
			}
			if ( ! empty( $this->cancel_url ) ) {
				$checkout_data['cancel_url'] = $this->cancel_url;
			}

			try {
				static::$CACHE[ $this->price_id ] = $this->stripe_connection->checkout->sessions->create( $checkout_data, [
					'stripe_account' => Connection::get_instance()->get_account_id(),
				] )->url;
			} catch ( Exception $e ) {

			}
		}

		return isset( static::$CACHE[ $this->price_id ] ) ? static::$CACHE[ $this->price_id ] : '';
	}

	/**
	 * Append the price_id and product_id to the URL
	 *
	 * @param $url
	 *
	 * @return string
	 */
	protected function append_url( $url ) {
		return add_query_arg( [
			'price_id'   => $this->price_id,
			'product_id' => $this->product_id,
		], $url );
	}
}
